\name{optgr}
\alias{optgr}
%- Also NEED an '\alias' for EACH other topic documented here.
\title{
  Minimize a function over a Grassmann manifold
}
\description{
  Minimizing a function over a Gr(n,p), using Newton's method if a Hessian is
  provided, and the steepest descent method otherwise.
}
\usage{
optgr(y0, fun, grEu, hessEu = NULL, maxit = 50, tol = 1e-8, ltol = 1e-3,
      alpha = 0.2, beta = 0.5)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{y0}{
    Starting point at the Grassmann manifold, an n \eqn{\times}{x} p
    matrix.
  }
  \item{fun}{ 
    The function to be minimized. \code{fun} should take an 
    n \eqn{\times}{x} p matrix as input and return a scalar.
  }
  \item{grEu}{
    The gradient of \code{fun} in Euclidean space. \code{grEu} should take an
    n \eqn{\times}{x} p matrix as input and return an n \eqn{\times}{x} p 
    matrix.
  }
  \item{hessEu}{
    The Hessian of \code{fun} in Euclidean space. \code{grEu} should take an
    n \eqn{\times}{x} p matrix as input and return an np \eqn{\times}{x} np 
    matrix.
  }
  \item{maxit}{
    Maximum number of iterations.
  }
  \item{tol}{
    Tolerated value for \eqn{\nabla f^T B^{-1} \nabla f}{(grad f)'B(grad f)}
    (Newton) or \eqn{||\nabla f||^2}{||grad f||^2} (steepest descent) at the
    optimal point.
  }
  \item{ltol}{
    Smallest eigenvalue of \eqn{B}.
  }
  \item{alpha}{
    Parameter for line search.
  }
  \item{beta}{
    Parameter for line search.
  }
}
\details{
  The Newton step is \eqn{v = \Pi(B^{-1}\nabla f)}{v = Pi(B^-1 grad f)}, where
  \eqn{\Pi}{Pi} is the projection onto the tangent plane, 
  \eqn{B = Hess(f) + \epsilon I}{B = Hess(f) + cI}, \eqn{Hess(f)} is a matrix
  interpretation of the Hessian of \code{fun} on Gr(n,p), \eqn{\epsilon}{c} is 
  chosen such that the smallest eigenvalue of \eqn{B} is \code{ltol} and
  \eqn{\nabla f}{grad f} is the gradient of \code{fun} along the Gr(n,p). For
  further information, see the reference and the package vignette.
  
  For the steepest descent method, the corresponding step is \eqn{-\nabla f}{
  grad f}, i.e.\ the negative of the gradient along Gr(n,p).
  
  For both methods, backtracking line search is used.  
}
\value{
  A list with components
  \item{value}{Value of \code{fun} at the found solution}
  \item{y}{A list with the points on the Grassmann manifold (matrices)
    that gone through during the iterations.}
  \item{lam2}{A list with the values of 
    \eqn{\nabla f^T B^{-1} \nabla f}{(grad f)'B(grad f)} (Newton) or 
    \eqn{||\nabla f||^2}{||grad f||^2} (steepest descent) at the points in
    \code{y}.}

}
\references{
  Edelman, A., Arias, T. A., and Smith S. T. (1998) The geometry of
  algorithms with orthogonality constraints. \emph{SIAM J. Matrix Anal. Appl.},
  \bold{20}(2), 303-353.
}
\author{
  Kerstin Johnsson, Lund University.
}
\note{
%%  ~~further notes~~
}

%% ~Make other sections like Warning with \section{Warning }{....} ~

\seealso{
\code{\link{ntStep}}, \code{\link{geod}}
}
\examples{
n <- 20
p <- 4

A <- matrix(runif(n^2), n, n)
A <- (A + t(A))/2

f1 <- function(y) sum(diag(t(y) \%*\% A \%*\% y))/2
f1y <- function(y) (A + t(A)) \%*\% y/2
f1yy <- function(y) {
  B <- matrix(0, n*p, n*p)
  for (j in 1:p) {
    B[((j-1)*n + 1):(j*n),((j-1)*n + 1):(j*n)] <- A
  }
  return(B)
}

y0 <- rbind(diag(1, p), matrix(0, n-p, p))
opt.res <- optgr(y0, f1, f1y, f1yy)

par(mfrow = c(1, 2))
plot(sapply(opt.res$y, f1), type = 'l', ylab = 'F(Y)', xlab = 'Iteration')
optval <- sum(sort(eigen(A, only.values = TRUE)$values)[1:p])/2
abline(h = optval, lty = 2)
print(optval)
err <- abs(sapply(opt.res$y, f1) - optval)
plot(err, log = 'y', ylab = 'Absolute error', xlab = 'Iteration')


}
% Add one or more standard keywords, see file 'KEYWORDS' in the
% R documentation directory.
%\keyword{ ~kwd1 }
%\keyword{ ~kwd2 }% __ONLY ONE__ keyword per line
